-- StopGQuit.lua - robust override + backup interceptor
local ADDON = "StopGQuit"
local frame = CreateFrame("Frame")
local tried = 0
local maxTries = 200       -- keep trying for a little while (OnUpdate)
local interval = 0.05      -- seconds between tries

-- Simple debug print; comment out if spammy
local function dbg(...) DEFAULT_CHAT_FRAME:AddMessage("|cffffff00"..ADDON..":|r "..strjoin(" ", tostringall(...))) end

-- Popup
StaticPopupDialogs["STOPGQUIT_CONFIRM"] = {
    text = "Are you sure you want to leave your guild? (Type YES to confirm)",
    button1 = "Yes",
    button2 = "No",
    hasEditBox = false,
    OnAccept = function()
        -- User clicked, so it's a real user action - call protected function
        GuildLeave()
    end,
    timeout = 0,
    whileDead = true,
    hideOnEscape = true,
}

-- Replacement slash handler
local function MyGQuitSlash()
    StaticPopup_Show("STOPGQUIT_CONFIRM")
end

-- Backup: intercept chat entry submit (to catch typed /gquit if slash override is stomped)
-- Save original if it exists
local origChatEdit_OnEnterPressed = ChatEdit_OnEnterPressed
local function BackupChatEdit_OnEnterPressed(self)
    local text = self:GetText()
    if text then
        local t = strlower(strmatch(text, "^%s*(%S+)") or "")
        if t == "/gquit" or t == "/guildquit" then
            -- show popup and clear the editbox to prevent the command from executing
            StaticPopup_Show("STOPGQUIT_CONFIRM")
            -- clear and defocus to stop submission
            self:HighlightText(0,0)
            self:SetText("")
            self:ClearFocus()
            dbg("Backup interceptor caught '"..text.."'. Showing confirmation.")
            return
        end
    end
    -- Not our command: call original
    if origChatEdit_OnEnterPressed then
        return origChatEdit_OnEnterPressed(self)
    end
end

-- Try to register/override the slash handler. Some addons / Blizzard re-register later,
-- so we attempt repeatedly with OnUpdate until we succeed or hit maxTries.
local elapsedAcc = 0
frame:SetScript("OnEvent", function(self, event, ...)
    if event == "PLAYER_LOGIN" then
        -- dbg("PLAYER_LOGIN received - attempting to register slash override and backup.")
        -- set the SLASH_ variables (helps registrations)
        SLASH_GQUIT1 = "/gquit"
        SLASH_GQUIT2 = "/guildquit"
        -- attempt once now
        SlashCmdList["GQUIT"] = MyGQuitSlash
        -- set backup chat interceptor now (replace function)
        if ChatEdit_OnEnterPressed ~= BackupChatEdit_OnEnterPressed then
            ChatEdit_OnEnterPressed = BackupChatEdit_OnEnterPressed
            -- dbg("Installed chat-entry backup interceptor.")
			print("\124cffFFD833GQuit confirmation dialog: \124cffFF0000 enabled\124r")
        end

        -- OnUpdate loop to re-apply the slash override in case something else overwrote it
        self:SetScript("OnUpdate", function(self, dt)
            elapsedAcc = elapsedAcc + dt
            if elapsedAcc < interval then return end
            elapsedAcc = 0
            tried = tried + 1
            local current = SlashCmdList and SlashCmdList["GQUIT"]
            if current ~= MyGQuitSlash then
                SlashCmdList["GQUIT"] = MyGQuitSlash
                SLASH_GQUIT1 = "/gquit"
                SLASH_GQUIT2 = "/guildquit"
                -- dbg("Re-applied SlashCmdList[\"GQUIT\"] (attempt "..tried..").")
            end
            -- if we've tried enough, stop the OnUpdate to avoid CPU usage
            if tried >= maxTries then
                -- dbg("Max tries reached ("..tostring(maxTries).."). Stopping retry loop.")
                self:SetScript("OnUpdate", nil)
            end
        end)
    end
end)
frame:RegisterEvent("PLAYER_LOGIN")
